--------------------------------------------------------------------
-- Login-Seite
--------------------------------------------------------------------

-- öffentlich zugängliche Login-Funktion
--DROP FUNCTION IF EXISTS api.login(text,text);
CREATE OR REPLACE FUNCTION api.login(
    username    text,
    password    text, -- als base64 inkl. Salt
    OUT token   text,
    OUT err_msg text
  )
  SECURITY DEFINER
  AS $$
  DECLARE
    _db_role    name;
    _ll_minr    integer;
  BEGIN
    -- decode password
    SELECT substring(
            convert_from( decode( password, 'base64' ), 'UTF-8' ),
            1,
            length( convert_from( decode( password, 'base64' ), 'UTF-8' ) ) - length( '.' )
            )
      INTO password;

    -- check user and password
    IF tpersonal.mitarbeiter__ll_db_usename__ll_passwd__password__check(username, password) THEN
      SELECT ll_minr
           , CASE ll_db_login
               WHEN true THEN ll_db_usename  -- Wenn User einen Prodat-Login hat dann seine die Datenbankrolle nehmen
               ELSE 'SYS.tools'              -- sonst allgemeine Rolle SYS.tools
             END AS db_role
        INTO _ll_minr
           , _db_role
        FROM llv
       WHERE ll_db_usename = username;
    ELSE
      RAISE invalid_password USING message = 'invalid user or password';
    END IF;

    IF NOT TSystem.rest__user_login__is_allowed( username => _db_role::varchar ) THEN
        err_msg := format('%s ist nicht zum REST-Login berechtigt. (Datenbank-Rolle: %s)', username, _db_role);
        RAISE WARNING '%', err_msg;
        RETURN;
    END IF;

    -- sign JWT
    SELECT x_10_interfaces.api_token_generate(
             -- _overwrite => true,
             _anbieter => 'PRODAT-ERP_POSTGREST',
             _user     => _db_role::varchar,
             _params   => jsonb_build_object('ll_minr', _ll_minr)
           )
      INTO token;
  END $$ LANGUAGE plpgsql;
  GRANT EXECUTE ON FUNCTION api.login(username text, password text) TO "sys_postgrest_anon";


-- Standard-HTML-Kopf für alle Seiten
CREATE OR REPLACE FUNCTION api.html_head_get(
    title varchar DEFAULT 'Prodat ERP'
  )
  RETURNS text
  AS $BODY$
    SELECT format(
      $HTML$
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <link rel="icon" href="./file?path=prodat_logo_pfeil.png" type="image/png">
        <title> %s </title>

        <!-- Tailwind CDN -->
        <script src="./file?path=extern/tailwindcss_3.4.16.js"></script>

        <!-- HTMX für dynamische Requests -->
        <script src="./file?path=extern/htmx.js"></script>

        <!-- HTMX Indicator Styles -->
        <style>
          .htmx-indicator { opacity: 0; }
          .htmx-request .htmx-indicator { opacity: 1; transition: opacity 200ms ease-in; }
          .htmx-request.htmx-indicator { opacity: 1; transition: opacity 200ms ease-in; }
        </style>

        <!-- Manifest -->
        <link rel="manifest" href="manifest" />

        <script type="module">
          import { getCookie, setupAuth, fetchWithAuth } from './file?path=auth-header.js';
          setupAuth(); // verwendet in der config im Standard => cookieName: 'jwt', headerName: 'Authorization', prefix: 'Bearer'
          window.fetchWithAuth = fetchWithAuth;

          // jwt aus Anfrage-Header als Cookie setzen wenn vorhanden
          let authorization_header = '%s' ;
          if (authorization_header && !getCookie('jwt')) {
            document.cookie = `jwt=${authorization_header}; path=/`;
          }

          // Login-Button anzeigen wenn kein gültiger Token gefunden wurde
          // Logout-Button anzeigen wenn Token vorhanden
          document.addEventListener('DOMContentLoaded', () => {
            const token = getCookie('jwt');
            const loginBtn = document.getElementById('loginBtn');
            const logoutBtn = document.getElementById('logoutBtn');
            if (!token && loginBtn) {
              loginBtn.style.display = 'inline-block'; // oder 'flex' je nach Layout
            } else if (token && logoutBtn) {
              logoutBtn.style.display = 'inline-block'; // oder 'flex' je nach Layout
            }
          });

          // Service-Worker für Offline-Funktionalität
          //if ('serviceWorker' in navigator) {
          //  navigator.serviceWorker.register('./file?id=inventur_service_worker.js')
          //    .then(reg => console.log("✅ Service Worker registriert:", reg.scope))
          //    .catch(err => console.error("❌ Fehler beim Registrieren:", err));
          //}
        </script>

        <script>
          window.F2Config = { moduln: 'TfrmToDesign' };  <!-- Prodat Modulname für F2 Suche -->
        </script>

        <!-- F2 Modal -->
        <script type="module">
          import {
            createF2Modal,
            initializeProdatInputs,
            configureF2Modal
          } from './file?path=f2_modal_tailwind.js';

          configureF2Modal({
            apiBase: '.', // optional anpassbar
            modalId: 'f2_modal' // falls mehrere benötigt
          });

          createF2Modal();
          initializeProdatInputs(); // oder mit selector: initializeProdatInputs('input[prodat^=\"X\"]');
        </script>
      </head>
      $HTML$, title, replace( current_setting('request.headers', true)::json->>'authorization', 'Bearer ', '' ) )
  $BODY$ LANGUAGE SQL IMMUTABLE;

/*
-- Standard-HTML-Kopf für alle Seiten
CREATE OR REPLACE FUNCTION api.html_head_get(
    title varchar DEFAULT 'PRODAT ERP'
    )
    RETURNS text
    AS $BODY$
      SELECT format(
      $HTML$
      <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <link rel="icon" href="./file?path=prodat_logo_pfeil.png" type="image/png">
        <title>%s</title>

        <!-- Tailwind CSS -->
        <script src="./file?path=extern/tailwindcss_3.4.16.js"></script>

        <!-- HTMX -->
        <script src="./file?path=extern/htmx.js"></script>

        <!-- HTMX Indicator Styles -->
        <style>
          .htmx-indicator { opacity: 0; }
          .htmx-request .htmx-indicator { opacity: 1; transition: opacity 200ms ease-in; }
        </style>

        <!-- Manifest -->
        <link rel="manifest" href="manifest" />

        <!-- Auth-Setup -->
        <script type="module">
          import { setupAuth } from './file?path=auth-header.js';
          setupAuth(); // setzt JWT-Cookie beim Erstaufruf
        </script>

        <script>
          window.F2Config = { moduln: 'TfrmToDesign' };
        </script>

        <!-- F2 Modal Initialisierung -->
        <script type="module">
          import {
            createF2Modal,
            initializeProdatInputs,
            configureF2Modal
          } from './file?path=f2_modal_tailwind.js';

          configureF2Modal({ apiBase: '.', modalId: 'f2_modal' });
          createF2Modal();
          initializeProdatInputs();
        </script>
      </head>
      $HTML$, title)
    $BODY$ LANGUAGE SQL IMMUTABLE;
*/


-- Funktion die den HTML-Code für die Login-Seite generiert
CREATE OR REPLACE FUNCTION api.login_page(
    redirect varchar DEFAULT 'dashboard_page'  -- Standard-Redirect nach erfolgreichem Login
  )
  RETURNS "text/html"
  SECURITY DEFINER
  AS $BODY$
    SELECT concat_ws( E'\n',
        '<!DOCTYPE html>', '<html lang="de">',
        api.html_head_get('Login'), format(
    $html$
      <body>
        <main class="max-w-md mx-auto mt-10 bg-white p-6 rounded-lg shadow">
          <article class="space-y-4">
            <img src="./file?path=prodat_logo.png" alt="Prodat Logo" class="mx-auto" />
            <h2 class="text-xl text-center font-semibold">Login</h2>
            <form id="loginForm" method="post" action="login" class="space-y-4">
              <input type="text" id="username" name="username" placeholder="Benutzername" required
                class="w-full px-4 py-2 border rounded" />
              <input type="password" id="password" name="password" placeholder="Passwort" required
                class="w-full px-4 py-2 border rounded" />
              <button type="submit"
                class="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 rounded">Login</button>
            </form>
            <section id="loginResult" class="text-center text-sm text-red-600"></section>

            <!-- JWT-Button & Dialog -->
            <button id="showJwtBtn" class="w-full bg-gray-100 border rounded py-2 mt-2 hidden">JWT anzeigen</button>
            <dialog id="jwtDialog" class="p-4 rounded shadow-xl">
              <h3 class="text-lg font-semibold mb-2">JWT-Details</h3>
              <div id="jwtInfo" class="bg-gray-50 rounded p-2 mb-2 text-xs overflow-x-auto"></div>
              <pre id="jwtPayload" class="bg-gray-50 rounded p-2 text-xs overflow-x-auto"></pre>
              <button id="closeJwtDialog" class="mt-4 bg-blue-600 text-white rounded px-4 py-2">Schließen</button>
            </dialog>
          </article>
        </main>

        <script>
          // Abfangen des Formular-Submits, um das Passwort zu hashen und vorhandenes Cookie zu löschen
          document.getElementById('loginForm').addEventListener('submit', async function(event) {
            event.preventDefault();

            document.cookie = "jwt=; path=/; expires=Thu, 01 Jan 1970 00:00:00 UTC;";

            const form = event.target;
            const passwordInput = form.password;
            const password = passwordInput.value;

            const salt = '.';
            const saltedPassword = password + salt;
            passwordInput.value = btoa(saltedPassword);

            // Formulardaten sammeln
            const formData = new FormData(form);
            const data = new URLSearchParams();
            for (const [key, value] of formData.entries()) {
              data.append(key, value);
            }

            try {
              const response = await fetch(form.action, {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: data
              });

              if (response.ok) {
                const result = await response.json();
                if (result.token) {
                  const payload = JSON.parse(base64UrlDecode(result.token.split('.')[1]));
                  const expirationDate = new Date(payload.exp * 1000);
                  const expires = expirationDate.toUTCString();
                  document.cookie = "jwt=" + result.token + "; path=/; expires=" + expires;
                  window.location.href = '%s'; // Redirect
                } else {
                  document.getElementById("loginResult").innerHTML = 'Unbekannte Antwort vom Server: ' + result.err_msg;
                }
              } else if (response.status === 403) {
                document.getElementById("loginResult").innerHTML = 'Login fehlgeschlagen. Bitte überprüfen Sie Ihre Eingaben.';
                const password = document.getElementById("password");
                if (password) {
                  password.setAttribute("aria-invalid", "true");
                }
              } else {
                document.getElementById("loginResult").innerHTML = 'Fehler beim Login-Vorgang.';
              }
            } catch (error) {
              document.getElementById("loginResult").innerHTML = 'Netzwerkfehler oder ungültige Antwort.';
            }
          });

          // ==== JWT-Button: Sichtbarkeit + Dialog ====

          // Hilfsfunktion: Hole JWT aus dem Cookie
          function getJwtFromCookie() {
            const match = document.cookie.match(/(?:^|; )jwt=([^;]+)/);
            return match ? match[1] : null;
          }

          // Zeige Button, wenn JWT existiert
          const jwtBtn = document.getElementById('showJwtBtn');
          const jwtDialog = document.getElementById('jwtDialog');
          const jwtPayload = document.getElementById('jwtPayload');
          const closeJwtDialog = document.getElementById('closeJwtDialog');

          function base64UrlDecode(str) {
            str = str.replace(/-/g, '+').replace(/_/g, '/');
            while (str.length %% 4) str += '=';
            try {
              return decodeURIComponent(atob(str).split('').map(function(c) {
                return '%%' + ('00' + c.charCodeAt(0).toString(16)).slice(-2);
              }).join(''));
            } catch (e) {
              return '';
            }
          }

          function formatTimestamp(ts) {
            if (!ts) return '—';
            // JWT exp ist in Sekunden seit Epoch
            const d = new Date(ts * 1000);
            return d.toLocaleString();
          }

          function showJwtDetails() {
            const jwt = getJwtFromCookie();
            const jwtInfo = document.getElementById('jwtInfo');
            if (!jwt) {
              jwtInfo.innerHTML = 'Kein gültiges JWT gefunden!';
              jwtPayload.textContent = '';
              jwtDialog.showModal();
              return;
            }
            const parts = jwt.split('.');
            if (parts.length !== 3) {
              jwtInfo.innerHTML = 'Kein gültiges JWT gefunden!';
              jwtPayload.textContent = '';
            } else {
              try {
                const payload = JSON.parse(base64UrlDecode(parts[1]));
                // Standard-Infos
                const domain = location.hostname;
                // Ablauf: bevorzugt exp aus Payload, alternativ leer
                const expires = payload.exp ? formatTimestamp(payload.exp) : '—';
                jwtInfo.innerHTML = `
                  <strong>Cookie-Name:</strong> jwt<br>
                  <strong>Domain:</strong> ${domain}<br>
                  <strong>Wert:</strong> <span class="break-all">${jwt}</span><br>
                  <strong>Ablauf (aus JWT):</strong> ${expires}
                `;
                jwtPayload.textContent = JSON.stringify(payload, null, 2);
              } catch (e) {
                jwtInfo.innerHTML = 'Fehler beim Dekodieren des JWT!';
                jwtPayload.textContent = '';
              }
            }
            jwtDialog.showModal();
          }

          // Sichtbarkeit prüfen beim Laden
          if (getJwtFromCookie()) {
            jwtBtn.classList.remove('hidden');
          }

          jwtBtn.addEventListener('click', showJwtDetails);
          closeJwtDialog.addEventListener('click', () => jwtDialog.close());
        </script>
      </body>
    </html>
    $html$, redirect )
    );
  $BODY$ LANGUAGE sql;
  GRANT EXECUTE ON FUNCTION api.login_page(varchar) TO "sys_postgrest_anon";


-- Umleitung zur Login-Seite
-- Diese Funktion wird aufgerufen, wenn der JWT-Token ungültig ist
-- und der Benutzer zur Login-Seite umgeleitet werden soll.
CREATE OR REPLACE FUNCTION api.html_redirect_get()
  RETURNS "text/html"
  AS $BODY$
    SELECT
      $html$<!DOCTYPE html>
      <html lang="de">
      <head>
          <meta charset="UTF-8">
          <title>Umleitung zur Startseite</title>
          <!-- Direkt nach 0 Sekunden umleiten -->
          <meta http-equiv="refresh" content="0; URL='login_page'" />
      </head>
      <body>
          <p>Sie werden weitergeleitet. Falls die Umleitung nicht funktioniert, klicken Sie bitte <a href="login_page">hier</a>.</p>
      </body>
      </html>
      $html$
  $BODY$ LANGUAGE SQL IMMUTABLE;